<?php

namespace App\Http\Controllers\API\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Auth;
use Socialite;
use Validator;
use App\Models\Vendor;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\HttpFoundation\Response;

class VendorAPIController extends Controller
{
    private $language;
    public function __construct()
    {
        // No need to call parent::__construct() here
        $this->language = request()->header('language-code') && request()->header('language-code') != '' ? request()->header('language-code') : 'en';
    }
    
    /**
     *  Do Signup
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    function doSignup(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:vendors',
            'phone' => ['required', 'unique:vendors', 'regex:/^\+\d{1,3}\d{6,14}$/'],
            'password' => 'required|min:8',
            'confirm_password' => 'required|same:password',
            'profile_pic' => 'image|max:2048',
            'device_type' => 'required|in:web,app',
        ]);

        if ($validator->fails()) {
            return $this->sendError(__('lang.message_required_message'),$validator->errors());
        }

        $vendorData = [
            'type' => 'vendor',
            'language_code' => 'en',
            'device_type' => $request->device_type,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'alternate_number' => $request->alternate_number,
            'role_id' => '3',
            'auth_token' => \Helpers::generateToken(),
            'status' => '1',
            'password' => Hash::make($request->confirm_password),
            'created_at' => date("Y-m-d H:i:s")
        ];

        if (isset($request['profile_pic']) && $request['profile_pic'] != '') {
            $uploadImage = \Helpers::uploadFiles($request['profile_pic'], 'vendor/profile_pic');
            if ($uploadImage['status'] == true) {
                $vendorData['image'] = $uploadImage['file_name'];
            }
        }

        $vendor_id = Vendor::insertGetId($vendorData);
        if ($vendor_id) {
            $vendorData = Vendor::where('id', $vendor_id)->first();
            if ($vendorData) {
                if ($vendorData->profile_pic != '') {
                    $vendorData->profile_pic = url('uploads/vendor/profile_pic/' . $vendorData->profile_pic); // Changed 'image' to 'profile_pic'
                }
            }
            return $this->sendResponse($vendorData, __('lang.message_signup_success'));
        }
        return $this->sendError(__('lang.message_user_registered'));
    }

    /**
     *  Do Sign In
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    function doLogin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email_phone' => 'required', // email or phone
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError(__('lang.message_required_message'), $validator->errors());
        }
        // Find user by email or phone
        $vendor = Vendor::where(function($query) use ($request) {
                $query->where('email', $request->email_phone)
                      ->orWhere('phone', $request->email_phone);
            })
            ->where('type', 'vendor')
            ->first();

        if (!$vendor) {
            return $this->sendError(__('lang.message_wrong_details'));
        }

        // Check if the password is correct
        if (!Hash::check($request->password, $vendor->password)) {
            return $this->sendError(__('lang.message_wrong_details'));
        }
        // Update the token
        $vendor->auth_token = \Helpers::generateToken();
        $vendor->save();

        // Return user data with updated token
        $vendor->profile_pic = $vendor->profile_pic ? url('uploads/vendor/profile_pic/' . $vendor->profile_pic) : null;

        return $this->sendResponse($vendor, __('lang.message_login_success'));
    }

}
