<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Hotel extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'hotels';

    public function additionalInfos()
    {
        return $this->hasMany(Additional_Info::class, 'hotel_id');
    }
    
    public function destination()
    {
        return $this->belongsTo(Destination::class, 'destination_id', 'id');
    }

    public function cancellationPolicies()
    {
        return $this->hasMany(Cancellation_Policy::class, 'hotel_id');
    }
    
    public function specification()
    {
        return $this->hasMany(Specification::class, 'hotel_id');
    }

    public function hotelAminities()
    {
        return $this->hasMany(Hotel_Aminities::class, 'hotel_id');
    }
    
    public function coupons()
    {
        return $this->belongsToMany(Coupon::class, 'coupon_hotels', 'hotel_id', 'coupon_id');
    }
    
    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }
    
    // Define the many-to-many relationship with packages
    public function packages()
    {
        return $this->belongsToMany(Package::class, 'hotel_packages', 'hotel_id', 'packages_id')
                    ->withPivot('price','guest', 'type', 'date')
                    ->withTimestamps();
    }


    public function hotelPackages()
    {
        return $this->hasMany(Hotel_Packages::class, 'hotel_id');
    }

    public function hotelImages()
    {
        return $this->hasMany(Hotel_Images::class, 'hotel_id');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function aminities()
    {
        return $this->belongsToMany(Aminity::class, 'hotel_aminities', 'hotel_id', 'aminities_id');
    }
    
    /**
     * Fetch list of from here
    **/
    public static function getLists($search)
    {
        try {
            $obj = new self;

            $pagination = (isset($search['perpage'])) ? $search['perpage'] : config('constant.pagination');

            if (isset($search['name']) && !empty($search['name'])) {
                $obj = $obj->where('name', 'like', '%' . trim($search['name']) . '%');
            }

            if (isset($search['status']) && $search['status'] !== '') {
                $obj = $obj->where('status', $search['status']);
            }

            if (isset($search['phone']) && $search['phone'] !== '') {
                $obj = $obj->where('phone', $search['phone']);
            }

            if (isset($search['destination_id']) && $search['destination_id'] !== '') {
                $obj = $obj->where('destination_id', $search['destination_id']);
            }
            
            if (isset($search['category_id']) && $search['category_id'] !== '') {
                $obj = $obj->where('category_id', $search['category_id']);
            }

            $data = $obj->with(['category', 'destination'])->orderBy('id', 'DESC')->paginate($pagination)->appends('perpage', $pagination);

            return $data;
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }
    
    /**
     * Add or update
    **/
    public static function addUpdate($data, $id = 0)
    {
        try {
            $obj = new self;
            $aminities_id_arr = $data['aminities_id'] ?? [];
            $additional_title_arr = $data['additional_title'] ?? [];
            $cancellation_policy_title_arr = $data['cancellation_policy_title'] ?? [];
            $specification_title_arr = $data['specification_title'] ?? [];
            $images_arr = $data['images'] ?? [];
            
            // Assuming $data contains the form data
            $packagesIdArr = $data['packages_id'] ?? [];
            $packagePriceArr = $data['package_price'] ?? [];
            $packageTypeArr = $data['package_type'] ?? [];
            $packageGuestArr = $data['package_guest'] ?? [];
            $packageDateArr = $data['package_date'] ?? [];

            unset($data['_token']);
            unset($data['aminities_id']);
            unset($data['additional_title']);
            unset($data['cancellation_policy_title']);
            unset($data['specification_title']);
            unset($data['packages_id']); 
            unset($data['package_price']);
            unset($data['package_type']);
            unset($data['package_guest']);
            unset($data['package_date']);
            
            unset($data['images']);
    
            // Handle image uploads
            if (isset($data['front_image']) && $data['front_image'] != '') {
                $uploadImage = \Helpers::uploadFiles($data['front_image'], 'hotel/front_image');
                if ($uploadImage['status'] == true) {
                    $data['front_image'] = $uploadImage['file_name'];
                }
            }
            if (isset($data['license_document']) && $data['license_document'] != '') {
                $uploadImage = \Helpers::uploadFiles($data['license_document'], 'hotel/license_documents');
                if ($uploadImage['status'] == true) {
                    $data['license_document'] = $uploadImage['file_name'];
                }
            }
    
            if ($id == 0) {
                $data['created_at'] = date('Y-m-d H:i:s');
                $hotel_id = $obj->insertGetId($data);
                
                // Handle amenities
                if(isset($aminities_id_arr) && $aminities_id_arr!=''){
                        foreach ($aminities_id_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'aminities_id' => $value
                            );
                            Hotel_Aminities::create($data);
                        }
                }
    
                // Handle additional info
                if(isset($additional_title_arr) && $additional_title_arr!=''){
                        foreach ($additional_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'title' => $value
                            );
                            Additional_Info::create($data);
                        }
                }
    
                // Handle cancellation policies
                if(isset($cancellation_policy_title_arr) && $cancellation_policy_title_arr!=''){
                        foreach ($cancellation_policy_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'title' => $value
                            );
                            Cancellation_Policy::create($data);
                        }
                }
    
                // Handle specifications
                if(isset($specification_title_arr) && $specification_title_arr!=''){
                        foreach ($specification_title_arr as $key => $value) {
                            $data = array(
                                'hotel_id'=> $hotel_id,
                                'title' => $value
                            );
                            Specification::create($data);
                        }
                }
    
                // Handle image uploads
                if (isset($images_arr) && !empty($images_arr)) {
                            foreach ($images_arr as $image) {
                        \Log::info('Uploading image: ', ['image' => $image->getClientOriginalName()]);
        
                        $uploadImage = \Helpers::multipleUploadFiles($image, 'hotel/images');
                        if ($uploadImage['status'] == true) {
                            Hotel_Images::create([
                                'hotel_id' => $hotel_id,
                                'images' => $uploadImage['file_name'],
                            ]);
                        } else {
                            \Log::error('Image upload failed: ', ['error' => $uploadImage['message']]);
                        }
                    }
                }
                
                // Check if arrays are not empty and have equal lengths
                if (!empty($packagesIdArr) && count($packagesIdArr) === count($packagePriceArr)
                    && count($packagesIdArr) === count($packageTypeArr)
                    && count($packagesIdArr) === count($packageGuestArr)
                    && count($packagesIdArr) === count($packageDateArr)) {
                
                    foreach ($packagesIdArr as $key => $packageId) {
                        $data = [
                            'hotel_id' => $hotel_id,
                            'packages_id' => $packageId,
                            'price' => $packagePriceArr[$key],
                            'type' => $packageTypeArr[$key],
                            'guest' => $packageGuestArr[$key],
                            'date' => $packageDateArr[$key]
                        ];
                
                        Hotel_Packages::create($data);
                    }
                }
                return ['status' => true, 'message' => __('lang.admin_data_add_msg')];
            } else {
                $data['updated_at'] = date('Y-m-d H:i:s');
                $obj->where('id',$id)->update($data);
    
                // Handle any deletion or update logic for existing records
    
                return ['status' => true, 'message' => __('lang.admin_data_update_msg')];
            }
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }
    
    /**
     * Delete particular
    **/
    public static function deleteRecord($id) {
    try {
        // Delete the record with the given id
        $obj = self::find($id);

        if ($obj) {
            $obj->delete();
            return ['status' => true, 'message' => __('lang.admin_data_delete_msg')];
        } else {
            return ['status' => false, 'message' => __('lang.admin_data_not_found')];
        }
    } catch (\Exception $e) {
        return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
    }
    }


}
