<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Model;
use DB;

class Package extends Model
{
    
    use HasFactory;
    use SoftDeletes;
    protected $guarded = ['id'];
    protected $table = "packages";
    
    public function features()
    {
        return $this->belongsTo(Feature::class, 'feature_id');
    }
    
    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }
    
    public function hotels()
    {
        return $this->belongsToMany(Hotel::class, 'hotel_packages', 'packages_id', 'hotel_id')
                    ->withPivot('price', 'type', 'date')
                    ->withTimestamps()
                    ->withSoftDeletes();
    }
    
    /**
     * Fetch list of from here
    **/
    public static function getLists($search){
        try {
            
            $obj = new self;

            $pagination = (isset($search['perpage']))?$search['perpage']:config('constant.pagination');

            if(isset($search['name']) && !empty($search['name'])){
                $obj = $obj->where('name', 'like', '%'.trim($search['name']).'%');
            } 

            if(isset($search['type']) && $search['type']!=''){
                $obj = $obj->where('type',$search['type']);
            } 

            if(isset($search['status']) && $search['status']!=''){
                $obj = $obj->where('status',$search['status']);
            } 
            
            if (isset($search['feature_id']) && !empty($search['feature_id'])) {
                $featureIds = is_array($search['feature_id']) ? $search['feature_id'] : explode(',', $search['feature_id']);
                
                $obj = $obj->where(function ($query) use ($featureIds) {
                    foreach ($featureIds as $featureId) {
                        $query->orWhereRaw('FIND_IN_SET(?, feature_id)', [$featureId]);
                    }
                });
            }

    
            $data = $obj->orderBy('id', 'DESC')->paginate($pagination)->appends('perpage', $pagination);

            return $data;
        }
        catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        }
    }

    /**
     * Add or update
    **/
    public static function addUpdate($data,$id=0) {
      
        try {
            $obj = new self;
            unset($data['_token']);
            $data['feature_id']=implode(",",$data['feature_id']);
            if($id==0){
                $data['created_at'] = date('Y-m-d H:i:s');
                $category_id = $obj->insertGetId($data);
                return ['status' => true, 'message' => __('lang.admin_data_add_msg')];
            }
            else{
                $data['updated_at'] = date('Y-m-d H:i:s');
                $obj->where('id',$id)->update($data);
                return ['status' => true, 'message' => __('lang.admin_data_update_msg')];
            }  
        }
        catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        }
    }

    /**
     * Delete particular
    **/
    public static function deleteRecord($id) {
        try {
            $obj = new self;    
            $obj->where('id',$id)->delete();   
            return ['status' => true, 'message' => __('lang.admin_data_delete_msg')];
        }
        catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        }
    }

    /**
     * Update Columns 
    **/
    public static function updateColumn($id){
        try {
            $data = self::where('id', $id)->first();

            // Assuming 'status' is an ENUM column with values '0' and '1'
            $newStatus = ($data->status == '1') ? '0' : '1';
            $data->update(['status' => $newStatus]);
            return ['status' => true, 'message' => __('lang.admin_data_change_msg')];
        } catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' ' . $e->getLine() . ' ' . $e->getFile()];
        }
    }


    /**
     * Fetch search item for pos
    **/
    public static function seachPosItems($search){
        try {
            
            $obj = new self;

            $pagination = (isset($search['perpage']))?$search['perpage']:config('constant.pagination');

            if(isset($search['name']) && !empty($search['name'])){
                $obj = $obj->where('name', 'like', '%'.trim($search['name']).'%');
            } 

            if(isset($search['restaurant_id']) && $search['restaurant_id']!=''){
                $obj = $obj->where('restaurant_id',$search['restaurant_id']);
            } 

            if(isset($search['category_id']) && $search['category_id']!=''){
                $obj = $obj->where('category_id',$search['category_id']);
            } 

            $data = $obj->with('category','restaurant')->orderBy('id', 'DESC')->paginate($pagination)->appends('perpage', $pagination);

            return $data;
        }
        catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        }
    }

    /**
     * Fetch cart items for pos
    **/
    public static function fetchCartItems($itemIds){
        try {
            
            $obj = new self;

            $pagination = (isset($search['perpage']))?$search['perpage']:config('constant.pagination');

            $data = $obj->whereIn('id',$itemIds)->with('category','restaurant')->orderBy('id', 'DESC')->paginate($pagination)->appends('perpage', $pagination);

            return $data;
        }
        catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        }
    }

}
